<?php
if (! defined('ABSPATH')) exit;

/**
 * Erzwingt nutzerbezogene Kategorien für die Taxonomie `booking_type`.
 * - Beim Anlegen eines Terms wird `owner_user_id` (Term‑Meta) gesetzt.
 * - Term-Abfragen werden für alle Benutzer so gefiltert, dass nur eigene Kategorien erscheinen.
 * - Bearbeiten/Löschen von fremden Kategorien wird unterbunden.
 */
class HBDEV_Budget_Tax_Ownership {
    const TAX = 'booking_type';
    const META_OWNER = 'owner_user_id';

    public static function init() {
        // Owner beim Erstellen setzen
        add_action('created_' . self::TAX, [__CLASS__, 'set_owner_on_create'], 10, 2);
        // Sichtbarkeit filtern (Admin & UI & Zuweisungs-Listen)
        add_action('pre_get_terms', [__CLASS__, 'filter_terms_for_owner']);
        // Bearbeiten/Löschen absichern
        add_action('edit_term', [__CLASS__, 'assert_owner_on_edit'], 10, 3);
        // Vor dem Löschen prüfen (Action, liefert 2 Parameter term_id, taxonomy)
        add_action('pre_delete_term', [__CLASS__, 'block_delete_if_not_owner'], 10, 2);
    }

    public static function set_owner_on_create($term_id, $tt_id) {
        $uid = get_current_user_id();
        if ($uid > 0) {
            // Nur einmal setzen, vorhandene Werte nicht überschreiben
            if (! get_term_meta($term_id, self::META_OWNER, true)) {
                add_term_meta($term_id, self::META_OWNER, (string) $uid, true);
            }
        }
    }

    /**
     * Filtert alle Term-Abfragen so, dass nur die Kategorien des aktuellen Nutzers erscheinen.
     */
    public static function filter_terms_for_owner($term_query) {
        if (! $term_query instanceof WP_Term_Query) return;
        $tax = $term_query->query_vars['taxonomy'] ?? [];
        // taxonomy kann string oder array sein
        $tax_list = is_array($tax) ? $tax : ( $tax ? [$tax] : [] );
        if (! in_array(self::TAX, $tax_list, true)) {
            return;
        }

        // Keine Ausnahmen: Filter gilt für Frontend, REST und Admin gleichermaßen –
        // jeder Benutzer sieht ausschließlich seine eigenen Begriffe.
        $uid = get_current_user_id();
        if ($uid <= 0) {
            // Für nicht eingeloggte keine Terms
            $term_query->query_vars['include'] = [0];
            return;
        }
        // Nur eigene Begriffe anzeigen (keine ownerlosen / fremden Begriffe sichtbar machen)
        $mq = $term_query->query_vars['meta_query'] ?? [];
        if (! is_array($mq)) $mq = [];
        $mq[] = [
            'key'     => self::META_OWNER,
            'value'   => (string) $uid,
            'compare' => '=',
        ];
        $term_query->query_vars['meta_query'] = $mq;
    }

    public static function assert_owner_on_edit($term_id, $tt_id, $taxonomy) {
        if ($taxonomy !== self::TAX) return;
        $uid = get_current_user_id();
        $owner = (int) get_term_meta($term_id, self::META_OWNER, true);
        if ($uid <= 0 || $owner !== $uid) {
            // Edit verhindern
            wp_die(esc_html__('Sie können diese Kategorie nicht bearbeiten, da sie nicht Ihnen gehört.', 'hbdev-budget'));
        }
    }

    public static function block_delete_if_not_owner($term_id, $taxonomy) {
        if ($taxonomy !== self::TAX) return;
        $uid = get_current_user_id();
        $owner = (int) get_term_meta($term_id, self::META_OWNER, true);
        if ($uid <= 0 || $owner !== $uid) {
            wp_die(esc_html__('Sie können diese Kategorie nicht löschen, da sie nicht Ihnen gehört.', 'hbdev-budget'));
        }
    }
}

HBDEV_Budget_Tax_Ownership::init();
