<?php
if ( ! defined('ABSPATH') ) exit;

class HBDEV_Budget_Subscriptions_UI {
	public static function init() {
		add_filter('post_row_actions', [__CLASS__, 'add_row_action'], 10, 2);
		add_action('post_submitbox_misc_actions', [__CLASS__, 'add_edit_button']);
		add_action('admin_post_hbdev_copy_subscription', [__CLASS__, 'handle_copy_action']);
		add_action('admin_notices', [__CLASS__, 'render_notices']);
		add_action('admin_menu', [__CLASS__, 'register_active_menu']);
	}

	public static function register_active_menu() {
		// Aktive Abos
		add_submenu_page(
			'edit.php?post_type=booking',
			__('Aktive Abos', 'hbdev-budget'),
			__('Aktive Abos', 'hbdev-budget'),
			'edit_subscriptions',
			'hbdev-active-subscriptions',
			[__CLASS__, 'render_active_subscriptions_page'],
			20
		);

		// Gekündigte/Beendete Abos
		add_submenu_page(
			'edit.php?post_type=booking',
			__('Gekündigte Abos', 'hbdev-budget'),
			__('Gekündigte Abos', 'hbdev-budget'),
			'edit_subscriptions',
			'hbdev-cancelled-subscriptions',
			[__CLASS__, 'render_cancelled_subscriptions_page'],
			21
		);
	}

	public static function render_active_subscriptions_page() {
		if ( ! current_user_can('edit_subscriptions') ) {
			wp_die(__('Kein Zugriff.', 'hbdev-budget'));
		}

		$tz = wp_timezone();
		$today = (new DateTimeImmutable('now', $tz))->format('Y-m-d');
		$subs = self::get_active_subscriptions($today);
		// Summen vorbereiten (monatlich und jährlich)
		$sum_annual = 0.0;
		if (!empty($subs)) {
			foreach ($subs as $sub) {
				$amount = (float) get_post_meta($sub->ID, 'subscription_amount', true);
				$repeat = (string) get_post_meta($sub->ID, 'subscription_repeat', true);
				$sum_annual += self::compute_annual_cost($amount, $repeat);
			}
		}
		$sum_monthly = $sum_annual / 12.0;
		?>
		<div class="wrap">
			<?php $icon = HBDEV_Budget_Helpers::get_page_dashicon_class('hbdev-active-subscriptions'); ?>
			<h1>
				<?php if ($icon): ?><span class="dashicons <?php echo esc_attr($icon); ?>" style="vertical-align:middle; margin-right:6px;"></span><?php endif; ?>
				<?php esc_html_e('Aktive Abos', 'hbdev-budget'); ?>
			</h1>
			<p class="description"><?php esc_html_e('Übersicht aller aktuell aktiven Abos. Beträge sind die hinterlegten Zyklus-Beträge; die jährlichen Kosten sind hochgerechnet.', 'hbdev-budget'); ?></p>
			<div class="notice notice-info" style="padding:12px; margin:12px 0;">
				<strong><?php esc_html_e('Summen:', 'hbdev-budget'); ?></strong>
				<div style="display:flex; flex-direction: column; gap:10px; margin-top:10px;">
					<div><?php esc_html_e('Summe monatlich', 'hbdev-budget'); ?>: &euro;&nbsp;<span style="font-weight:600;"><?php echo self::fmt_eur($sum_monthly); ?></span></div>
					<div><?php esc_html_e('Summe jährlich', 'hbdev-budget'); ?>: &euro;&nbsp;<span style="font-weight:600;"><?php echo self::fmt_eur($sum_annual); ?></span></div>
				</div>
			</div>
			<table class="widefat striped">
				<thead>
				<tr>
					<th><?php esc_html_e('Titel', 'hbdev-budget'); ?></th>
					<th style="text-align:right; width:120px;">&euro; <?php esc_html_e('Betrag', 'hbdev-budget'); ?></th>
					<th style="width:120px;">&nbsp;<?php esc_html_e('Wiederholung', 'hbdev-budget'); ?></th>
					<th style="text-align:right; width:160px;">&euro; <?php esc_html_e('Jahreskosten', 'hbdev-budget'); ?></th>
					<th style="width:120px;"><?php esc_html_e('Start', 'hbdev-budget'); ?></th>
					<th style="width:120px;"><?php esc_html_e('Ende', 'hbdev-budget'); ?></th>
					<th><?php esc_html_e('Kategorien', 'hbdev-budget'); ?></th>
					<th style="width:220px;">&nbsp;</th>
				</tr>
				</thead>
				<tbody>
				<?php if (empty($subs)) : ?>
					<tr><td colspan="8"><?php esc_html_e('Keine aktiven Abos gefunden.', 'hbdev-budget'); ?></td></tr>
				<?php else: foreach ($subs as $sub) :
					$amount = (float) get_post_meta($sub->ID, 'subscription_amount', true);
					$repeat = (string) get_post_meta($sub->ID, 'subscription_repeat', true);
					$start  = (string) get_post_meta($sub->ID, 'subscription_start', true);
					$end    = (string) get_post_meta($sub->ID, 'subscription_end', true);
					$annual = self::compute_annual_cost($amount, $repeat);
					$cats   = self::get_categories_list($sub->ID);
					$copy_url = '';
					if ( current_user_can('publish_bookings') ) {
						$copy_url = wp_nonce_url( admin_url('admin-post.php?action=hbdev_copy_subscription&subscription_id=' . (int)$sub->ID), 'hbdev_copy_subscription_' . (int)$sub->ID );
					}
				?>
					<tr>
						<td><a href="<?php echo esc_url( get_edit_post_link($sub->ID) ); ?>"><?php echo esc_html( get_the_title($sub) ?: ('#' . (int)$sub->ID) ); ?></a></td>
						<td style="text-align:right;">&euro;&nbsp;<?php echo self::fmt_eur($amount); ?></td>
						<td><?php echo esc_html( self::label_repeat($repeat) ); ?></td>
						<td style="text-align:right; font-weight:600;">&euro;&nbsp;<?php echo self::fmt_eur($annual); ?></td>
						<td><?php echo esc_html($start); ?></td>
						<td><?php echo esc_html($end); ?></td>
						<td><?php echo wp_kses_post($cats); ?></td>
						<td>
							<?php if ($copy_url) : ?>
								<a class="button button-primary" href="<?php echo esc_url($copy_url); ?>" aria-label="<?php echo esc_attr(sprintf(__('Abo "%s" in laufenden Monat übernehmen', 'hbdev-budget'), get_the_title($sub))); ?>" onclick="return confirm('<?php echo esc_js(__('Soll dieses Abo als Buchung in den laufenden Monat übernommen werden?', 'hbdev-budget')); ?>');">
									<?php esc_html_e('In laufenden Monat übernehmen', 'hbdev-budget'); ?>
								</a>
							<?php else: ?>
								<em><?php esc_html_e('Keine Berechtigung für Buchungserstellung', 'hbdev-budget'); ?></em>
							<?php endif; ?>
						</td>
					</tr>
				<?php endforeach; endif; ?>
				</tbody>
			</table>
		</div>
		<?php
	}

 private static function get_active_subscriptions(string $today): array {
		$author_id = get_current_user_id();
		$meta_query = [
			'relation' => 'AND',
			[
				'key' => 'subscription_status',
				'value' => 'running',
				'compare' => '=',
			],
			[
				'relation' => 'AND',
				[
					'key' => 'subscription_start',
					'value' => $today,
					'compare' => '<=',
					'type' => 'DATE',
				],
				[
					'relation' => 'OR',
					[
						'key' => 'subscription_end',
						'compare' => 'NOT EXISTS',
					],
					[
						'key' => 'subscription_end',
						'value' => '',
						'compare' => '=',
					],
					[
						'key' => 'subscription_end',
						'value' => $today,
						'compare' => '>=',
						'type' => 'DATE',
					],
				],
			],
		];
		$q = new WP_Query([
			'post_type' => 'subscription',
			'post_status' => 'any',
			'posts_per_page' => 200,
			'orderby' => 'title',
			'order' => 'ASC',
			'author' => $author_id,
			'meta_query' => $meta_query,
		]);
		return $q->have_posts() ? $q->posts : [];
	}

	private static function compute_annual_cost(float $amount, string $repeat): float {
		switch ($repeat) {
			case 'weekly':
				$factor = 52;
				break;
			case 'yearly':
				$factor = 1;
				break;
			case 'monthly':
			default:
				$factor = 12;
		}
		return $amount * $factor;
	}

	private static function label_repeat(string $repeat): string {
		return $repeat === 'weekly' ? __('wöchentlich', 'hbdev-budget') : ($repeat === 'yearly' ? __('jährlich', 'hbdev-budget') : __('monatlich', 'hbdev-budget'));
	}

	private static function fmt_eur(float $amount): string {
		$formatted = number_format_i18n($amount, 2);
		return esc_html($formatted);
	}

	private static function get_categories_list(int $post_id): string {
		$terms = get_the_terms($post_id, 'booking_type');
		if (is_wp_error($terms) || empty($terms)) return '';
		$names = array_map(function($t){ return esc_html($t->name); }, $terms);
		return implode(', ', $names);
	}

	public static function render_cancelled_subscriptions_page() {
		if ( ! current_user_can('edit_subscriptions') ) {
			wp_die(__('Kein Zugriff.', 'hbdev-budget'));
		}

		$tz = wp_timezone();
		$today = (new DateTimeImmutable('now', $tz))->format('Y-m-d');
		$subs = self::get_cancelled_subscriptions($today);
		?>
		<div class="wrap">
			<?php $icon = HBDEV_Budget_Helpers::get_page_dashicon_class('hbdev-cancelled-subscriptions'); ?>
			<h1>
				<?php if ($icon): ?><span class="dashicons <?php echo esc_attr($icon); ?>" style="vertical-align:middle; margin-right:6px;"></span><?php endif; ?>
				<?php esc_html_e('Gekündigte Abos', 'hbdev-budget'); ?>
			</h1>
			<p class="description"><?php esc_html_e('Abos, die gekündigt wurden oder deren Enddatum erreicht/überschritten ist.', 'hbdev-budget'); ?></p>
			<table class="widefat striped">
				<thead>
				<tr>
					<th><?php esc_html_e('Titel', 'hbdev-budget'); ?></th>
					<th style="text-align:right; width:120px;">&euro; <?php esc_html_e('Betrag', 'hbdev-budget'); ?></th>
					<th style="width:120px;">&nbsp;<?php esc_html_e('Wiederholung', 'hbdev-budget'); ?></th>
					<th style="text-align:right; width:160px;">&euro; <?php esc_html_e('Jahreskosten', 'hbdev-budget'); ?></th>
					<th style="width:120px;">&nbsp;<?php esc_html_e('Start', 'hbdev-budget'); ?></th>
					<th style="width:120px;">&nbsp;<?php esc_html_e('Ende', 'hbdev-budget'); ?></th>
					<th><?php esc_html_e('Kategorien', 'hbdev-budget'); ?></th>
				</tr>
				</thead>
				<tbody>
				<?php if (empty($subs)) : ?>
					<tr><td colspan="7"><?php esc_html_e('Keine gekündigten oder beendeten Abos gefunden.', 'hbdev-budget'); ?></td></tr>
				<?php else: foreach ($subs as $sub) :
					$amount = (float) get_post_meta($sub->ID, 'subscription_amount', true);
					$repeat = (string) get_post_meta($sub->ID, 'subscription_repeat', true);
					$start  = (string) get_post_meta($sub->ID, 'subscription_start', true);
					$end    = (string) get_post_meta($sub->ID, 'subscription_end', true);
					$annual = self::compute_annual_cost($amount, $repeat);
					$cats   = self::get_categories_list($sub->ID);
				?>
					<tr>
						<td><a href="<?php echo esc_url( get_edit_post_link($sub->ID) ); ?>"><?php echo esc_html( get_the_title($sub) ?: ('#' . (int)$sub->ID) ); ?></a></td>
						<td style="text-align:right;">&euro;&nbsp;<?php echo self::fmt_eur($amount); ?></td>
						<td><?php echo esc_html( self::label_repeat($repeat) ); ?></td>
						<td style="text-align:right; font-weight:600;">&euro;&nbsp;<?php echo self::fmt_eur($annual); ?></td>
						<td><?php echo esc_html($start); ?></td>
						<td><?php echo esc_html($end); ?></td>
						<td><?php echo wp_kses_post($cats); ?></td>
					</tr>
				<?php endforeach; endif; ?>
				</tbody>
			</table>
		</div>
		<?php
	}

 private static function get_cancelled_subscriptions(string $today): array {
		$author_id = get_current_user_id();
		$meta_query = [
			'relation' => 'OR',
			[
				'key' => 'subscription_status',
				'value' => 'cancelled',
				'compare' => '=',
			],
			[
				'relation' => 'AND',
				[
					'key' => 'subscription_end',
					'compare' => 'EXISTS',
				],
				[
					'key' => 'subscription_end',
					'value' => '',
					'compare' => '!=',
				],
				[
					'key' => 'subscription_end',
					'value' => $today,
					'compare' => '<=',
					'type' => 'DATE',
				],
			],
		];

		$q = new WP_Query([
			'post_type' => 'subscription',
			'post_status' => 'any',
			'posts_per_page' => 200,
			'orderby' => 'title',
			'order' => 'ASC',
			'author' => $author_id,
			'meta_query' => $meta_query,
		]);
		return $q->have_posts() ? $q->posts : [];
	}

	public static function add_row_action($actions, $post) {
		if ($post->post_type !== 'subscription') return $actions;
		if (! current_user_can('publish_bookings')) return $actions;
		$url = wp_nonce_url( admin_url('admin-post.php?action=hbdev_copy_subscription&subscription_id=' . (int)$post->ID), 'hbdev_copy_subscription_' . (int)$post->ID );
		$actions['hbdev-copy-sub'] = '<a href="' . esc_url($url) . '">' . esc_html__('In laufenden Monat übernehmen', 'hbdev-budget') . '</a>';
		return $actions;
	}

	public static function add_edit_button() {
		global $post;
		if (! $post || $post->post_type !== 'subscription') return;
		if (! current_user_can('publish_bookings')) return;
		$url = wp_nonce_url( admin_url('admin-post.php?action=hbdev_copy_subscription&subscription_id=' . (int)$post->ID), 'hbdev_copy_subscription_' . (int)$post->ID );
		echo '<div class="misc-pub-section">
			<a class="button" href="' . esc_url($url) . '">' . esc_html__('In laufenden Monat übernehmen', 'hbdev-budget') . '</a>
		</div>';
	}

	public static function handle_copy_action() {
		$sub_id = isset($_GET['subscription_id']) ? (int) $_GET['subscription_id'] : 0;
		if (! $sub_id || ! wp_verify_nonce($_GET['_wpnonce'] ?? '', 'hbdev_copy_subscription_' . $sub_id)) {
			self::add_notice('error', __('Aktion nicht erlaubt.', 'hbdev-budget'));
			wp_safe_redirect( wp_get_referer() ?: admin_url('edit.php?post_type=subscription') );
			exit;
		}
		$subscription = get_post($sub_id);
		if (! $subscription || $subscription->post_type !== 'subscription') {
			self::add_notice('error', __('Abo nicht gefunden.', 'hbdev-budget'));
			wp_safe_redirect( wp_get_referer() ?: admin_url('edit.php?post_type=subscription') );
			exit;
		}
		if (! current_user_can('publish_bookings')) {
			self::add_notice('error', __('Keine Berechtigung, Buchungen zu erstellen.', 'hbdev-budget'));
			wp_safe_redirect( wp_get_referer() ?: admin_url('edit.php?post_type=subscription') );
			exit;
		}

		$amount = (float) get_post_meta($sub_id, 'subscription_amount', true);
		$note   = (string) get_post_meta($sub_id, 'subscription_note', true);
		$repeat = (string) get_post_meta($sub_id, 'subscription_repeat', true);
		$status = (string) get_post_meta($sub_id, 'subscription_status', true);
		$end    = (string) get_post_meta($sub_id, 'subscription_end', true);
		$start  = (string) get_post_meta($sub_id, 'subscription_start', true);

		$tz = wp_timezone();
		$today = new DateTimeImmutable('now', $tz);
		$ym = $today->format('Y-m');
		$booking_date = $today->format('Y-m-d');

		// Optionale einfache Validierung hinsichtlich Zeitraum
		$active_ok = true;
		if (preg_match('/^\d{4}-\d{2}-\d{2}$/', $start)) {
			$active_ok = $active_ok && ($booking_date >= $start);
		}
		if ($end && preg_match('/^\d{4}-\d{2}-\d{2}$/', $end)) {
			$active_ok = $active_ok && ($booking_date <= $end);
		}
		if ($status === 'cancelled') {
			$active_ok = false;
		}

		$title = get_the_title($subscription);
		$booking_title = sprintf(__('Abo: %s', 'hbdev-budget'), $title);

		// Erstelle Buchung
		$booking_id = wp_insert_post([
			'post_type'   => 'booking',
			'post_status' => 'publish',
			'post_title'  => $booking_title,
			'post_content'=> $note,
		]);
		if (is_wp_error($booking_id) || ! $booking_id) {
			self::add_notice('error', __('Buchung konnte nicht erstellt werden.', 'hbdev-budget'));
			wp_safe_redirect( wp_get_referer() ?: admin_url('edit.php?post_type=subscription') );
			exit;
		}

		update_post_meta($booking_id, 'booking_amount', $amount);
		update_post_meta($booking_id, 'booking_date', $booking_date);

		// Kategorien vom Abo übernehmen
		$terms = wp_get_object_terms($sub_id, 'booking_type', ['fields' => 'ids']);
		if (! is_wp_error($terms) && ! empty($terms)) {
			wp_set_object_terms($booking_id, $terms, 'booking_type', false);
		}

		self::add_notice('success', sprintf(__('Abo in %s übernommen. Neue Buchung #%d.', 'hbdev-budget'), esc_html($ym), (int)$booking_id));
		wp_safe_redirect( admin_url('post.php?action=edit&post=' . (int)$booking_id) );
		exit;
	}

	private static function add_notice($type, $message) {
		$stack = get_transient('hbdev_budget_notices');
		if (!is_array($stack)) $stack = [];
		$stack[] = ['type'=>$type, 'message'=>$message];
		set_transient('hbdev_budget_notices', $stack, 60);
	}

	public static function render_notices() {
		$stack = get_transient('hbdev_budget_notices');
		if (!is_array($stack) || empty($stack)) return;
		delete_transient('hbdev_budget_notices');
		foreach ($stack as $n) {
			$type = in_array($n['type'], ['success','error','warning','info'], true) ? $n['type'] : 'info';
			echo '<div class="notice notice-' . esc_attr($type) . ' is-dismissible"><p>' . wp_kses_post($n['message']) . '</p></div>';
		}
	}
}

add_action('admin_menu', function() {
	global $submenu;
	$parent = 'edit.php?post_type=booking';
	if (empty($submenu[$parent])) return;

	$activeIdx = null; $cancelIdx = null; $subIdx = null;
	foreach ($submenu[$parent] as $idx => $item) {
		// $item: [0 => Title, 1 => Cap, 2 => Slug]
		$slug = $item[2] ?? '';
		if ($slug === 'hbdev-active-subscriptions') { $activeIdx = $idx; }
		if ($slug === 'hbdev-cancelled-subscriptions') { $cancelIdx = $idx; }
		if ($slug === 'edit.php?post_type=subscription') { $subIdx = $idx; }
	}

	if ($activeIdx === null) return;

	// 1) "Gekündigte Abos" direkt hinter "Aktive Abos" anordnen (falls vorhanden)
	if ($cancelIdx !== null && isset($submenu[$parent][$cancelIdx])) {
		$cancelItem = $submenu[$parent][$cancelIdx];
		unset($submenu[$parent][$cancelIdx]);
		$submenu[$parent] = array_values($submenu[$parent]);

		// Aktive Abos erneut finden (Index kann sich verschoben haben)
		$activeIdx = null;
		foreach ($submenu[$parent] as $idx => $item) {
			if (($item[2] ?? '') === 'hbdev-active-subscriptions') { $activeIdx = $idx; break; }
		}
		if ($activeIdx !== null) {
			array_splice($submenu[$parent], $activeIdx + 1, 0, [$cancelItem]);
		}
	}

	// 2) CPT-Eintrag "Abos erstellen/bearbeiten" direkt hinter "Gekündigte Abos" (falls vorhanden),
	//    sonst direkt hinter "Aktive Abos" einsortieren
	if ($subIdx !== null && isset($submenu[$parent][$subIdx])) {
		$subscriptionItem = $submenu[$parent][$subIdx];
		unset($submenu[$parent][$subIdx]);
		$submenu[$parent] = array_values($submenu[$parent]);

		// Zielposition bestimmen: bevorzugt hinter "Gekündigte Abos"
		$targetIdx = null;
		foreach ($submenu[$parent] as $idx => $item) {
			$slug = $item[2] ?? '';
			if ($slug === 'hbdev-cancelled-subscriptions') { $targetIdx = $idx; break; }
		}
		if ($targetIdx === null) {
			foreach ($submenu[$parent] as $idx => $item) {
				if (($item[2] ?? '') === 'hbdev-active-subscriptions') { $targetIdx = $idx; break; }
			}
		}
		if ($targetIdx !== null) {
			array_splice($submenu[$parent], $targetIdx + 1, 0, [$subscriptionItem]);
		}
	}
}, 999);

HBDEV_Budget_Subscriptions_UI::init();
